package com.tyro.androidtta

import android.webkit.WebView
import org.hamcrest.CoreMatchers.`is`
import org.junit.Assert.assertThat
import org.junit.Before
import org.junit.Test
import org.junit.runner.RunWith
import org.mockito.Answers
import org.mockito.ArgumentMatchers
import org.mockito.Mock
import org.mockito.Mockito.`when`
import org.mockito.Mockito.verify
import org.mockito.junit.MockitoJUnitRunner
import java.util.concurrent.CountDownLatch
import java.util.concurrent.TimeUnit

@RunWith(MockitoJUnitRunner::class)
class TyroClientTest {

    @Mock(answer = Answers.RETURNS_DEEP_STUBS)
    private lateinit var webView: WebView

    private lateinit var tyroClient: TyroClient

    @Before
    fun setup() {
        tyroClient = TyroClient(
                webView,
                "someApiKey",
                PosProductData("ACME", "POS", "1.0.0"),
                IclientSource.SIMULATOR
        )
    }

    @Test
    fun `should configure web view correctly`() {
        verify(webView).addJavascriptInterface(tyroClient, "EmbeddedResponseAdapter")
        verify(webView.settings).javaScriptEnabled = true
        verify(webView.settings).domStorageEnabled = true
    }

    @Test
    fun `should load correct URL for pairTerminal`() {
        tyroClient.pairTerminal()
        verify(webView).loadUrl("${IclientSource.SIMULATOR.url}/configuration.html")
    }

    @Test
    fun `should perform operation with correct API Key and method arguments`() {
        `when`(webView.post(ArgumentMatchers.any())).thenAnswer {
            (it.arguments[0] as Runnable).run()
            true
        }

        tyroClient.performOperation(
                "someOperation",
                mapOf(Pair("paramA", "valueA"), Pair("paramB", "valueB")),
                { },
                { }
        )

        tyroClient.ready()

        verify(webView).loadUrl("${IclientSource.SIMULATOR.url}/embedded.html?apiKey=someApiKey")
        verify(webView).evaluateJavascript("""iClient.someOperation({"paramA":"valueA","paramB":"valueB","posProductData":{"posProductName":"POS","posProductVendor":"ACME","posProductVersion":"1.0.0"}})""".trimMargin(), null)
    }

    @Test
    fun `should invoke receipt callback`() {
        val latch = CountDownLatch(1)
        var receiptData: Map<String, String> = mapOf()

        tyroClient.performOperation(
                "someOperation",
                mapOf(Pair("paramA", "valueA"), Pair("paramB", "valueB")),
                {
                    receiptData = it
                    latch.countDown()
                },
                { }
        )

        tyroClient.receiptReceived("""{"data":"value"}""")
        latch.await(5, TimeUnit.SECONDS)
        assertThat(receiptData["data"], `is`("value"))
    }

    @Test
    fun `should invoke transaction complete callback`() {
        val latch = CountDownLatch(1)
        var transactionComplete: Map<String, String> = mapOf()

        tyroClient.performOperation(
                "someOperation",
                mapOf(Pair("paramA", "valueA"), Pair("paramB", "valueB")),
                { },
                {
                    transactionComplete = it
                    latch.countDown()
                }
        )

        tyroClient.transactionComplete("""{"data":"value"}""")
        latch.await(5, TimeUnit.SECONDS)
        assertThat(transactionComplete["data"], `is`("value"))
    }
}