//
//  PurchaseTest.swift
//  PosSimulator
//
//  Created by development on 12/22/15.
//  Copyright © 2015 Tyro Payment Limited. All rights reserved.
//

import Foundation
import Nimble

class PurchaseTest: TestBase {
    
    func testPurchase() {
        configurePos()
        performPurchase()
    }
    
    func testPurchaseWithBaseUrlContainingTrailingSlash() {
        configurePos(baseUrl: DEFAULT_URL + "/")
        performPurchase()
    }

    func testPurchaseWithIntegratedReceipt() {
        configurePos(andIntegratedReceipt: true)
        performPurchase(expectCustomerCopy: true, expectMerchantCopy: true)
    }


    
    func testPurchaseWithCashout() {
        configurePos()
        performPurchase(withPurchaseAmount: "10100", withCashOut: "1000")
    }
    
    func performPurchase(expectApproved: Bool = true, expectCustomerCopy: Bool = false, expectMerchantCopy: Bool = false,
                            withPurchaseAmount purchaseAmount: String = "10200", withCashOut cashOutAmount: String = "") {
        let paymentsScreen = payments()
            .purchaseAmount(purchaseAmount)
            .cashOutAmount(cashOutAmount)
            .initiatePurchase()
        
        if (!expectCustomerCopy) {
            expect(paymentsScreen.app.staticTexts["APPROVED. Print customer copy?"].exists).toEventually(beTrue())
            _ = paymentsScreen.no()
        }
        
        expect(paymentsScreen.app.staticTexts["APPROVED"].exists).toEventually(equal(expectApproved))
        expect((paymentsScreen.app.textViews["customerReceipt"].value as? String)?.containsString("CUSTOMER COPY")).toEventually(equal(expectCustomerCopy))
        expect((paymentsScreen.app.textViews["merchantReceipt"].value as? String)?.containsString("MERCHANT COPY")).toEventually(equal(expectMerchantCopy))
        expect((paymentsScreen.app.textViews["merchantReceipt"].value as? String)?.containsString("Surcharge")).toEventually(beFalse())
    }
    
    func testRefund() {
        configurePos()
        performRefund()
    }
    
    func testRefundWithIntegratedReciept() {
        configurePos(andIntegratedReceipt: true)
        performRefund(expectCustomerCopy: true, expectMerchantCopy: true)
    }
    
    func performRefund(expectApproved: Bool = true, expectCustomerCopy: Bool = false, expectMerchantCopy: Bool = false) {
        let paymentsScreen = payments()
            .refundAmount("10200")
            .refund()
        
        if (!expectCustomerCopy) {
            expect(paymentsScreen.app.staticTexts["APPROVED. Print customer copy?"].exists).toEventually(beTrue())
            _ = paymentsScreen.no()
        }
        
        expect(paymentsScreen.app.staticTexts["REFUND"].exists).toEventually(beTruthy())
        expect(paymentsScreen.app.staticTexts["APPROVED"].exists).toEventually(equal(expectApproved))
        expect((paymentsScreen.app.textViews["customerReceipt"].value as? String)?.containsString("CUSTOMER COPY")).toEventually(equal(expectCustomerCopy))
        expect((paymentsScreen.app.textViews["merchantReceipt"].value as? String)?.containsString("MERCHANT COPY")).toEventually(equal(expectMerchantCopy))
    }
    
    func testCancelPayment() {
        configurePos()
        let paymentsScreen = payments()
            .purchaseAmount("6001")
            .initiatePurchase()
        
        expect(paymentsScreen.app.staticTexts["Swipe card. Purchase: $60.01"].exists).toEventually(equal(true))
        
        app.buttons["\u{e020} Cancel Transaction"].tap()
        expect(paymentsScreen.app.staticTexts["Are you sure you want to cancel?"].exists).toEventually(equal(true))
        
        _ = paymentsScreen.yes()
        expect(paymentsScreen.app.staticTexts["CANCELLED"].exists).toEventually(equal(true))
    }
    
    func testPurchaseWithSurcharge() {
        configurePos(andIntegratedReceipt: true)
        
        app.switches["enableSurchargeSwitch"].tap()
        
        let paymentsScreen = payments()
            .purchaseAmount("10250")
            .initiatePurchase()
        
        expect((paymentsScreen.app.textViews["merchantReceipt"].value as? String)?.containsString("Surcharge    AUD       $2.05")).toEventually(beTrue(), timeout: 10)
    }
    
    func testContinueLastTransaction() {
        
        configurePos(andIntegratedReceipt: true)
        performPurchase(expectCustomerCopy: true, expectMerchantCopy: true)
        
        expect(self.app.buttons["Dismiss"].exists).toEventually(beTrue(), timeout:10)
        app.buttons["Dismiss"].tap()

        
        expect((self.app.textViews["customerReceipt"].value as? String)?.containsString("CUSTOMER COPY")).toEventually(beTrue())
        expect((self.app.textViews["merchantReceipt"].value as? String)?.containsString("MERCHANT COPY")).toEventually(beTrue())
        
        app.buttons["Clear POS"].tap()
        _ = payments().continueLastTransaction()
        
        expect((self.app.textViews["customerReceipt"].value as? String)?.containsString("CUSTOMER COPY")).toEventually(beTrue())
        expect((self.app.textViews["merchantReceipt"].value as? String)?.containsString("MERCHANT COPY")).toEventually(beTrue())
    }
}
